package org.eclipse.swt.widgets;

/*
 * OS/2 version.
 * Copyright (c) 2002, 2008 EclipseOS2 Team.
 */

/*
 * Copyright (c) 2000, 2002 IBM Corp.  All rights reserved.
 * This file is made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 */
 
import org.eclipse.swt.internal.*;
import org.eclipse.swt.internal.pm.*;
import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

/**
 * Instances of this class implement the notebook user interface
 * metaphor.  It allows the user to select a notebook page from
 * set of pages.
 * <p>
 * The item children that may be added to instances of this class
 * must be of type <code>TabItem</code>.
 * <code>Control</code> children are created and then set into a
 * tab item using <code>TabItem#setControl</code>.
 * </p><p>
 * Note that although this class is a subclass of <code>Composite</code>,
 * it does not make sense to set a layout on it.
 * </p><p>
 * <dl>
 * <dt><b>Styles:</b></dt>
 * <dd>(none)</dd>
 * <dt><b>Events:</b></dt>
 * <dd>Selection</dd>
 * </dl>
 * <p>
 * IMPORTANT: This class is <em>not</em> intended to be subclassed.
 * </p>
 */
public class TabFolder extends Composite {
	TabItem [] items;
        int[] itemsID;
        int iLongestMajText = 0;
//	ImageList imageList;
	static final int TabFolderProc;
	static final PSZ TabFolderClass = PSZ.getAtom (OS.WC_NOTEBOOK);
	static {
		
		/*
		* Feature in Windows.  The tab control window class
		* uses the CS_HREDRAW and CS_VREDRAW style bits to
		* force a full redraw of the control and all children
		* when resized.  This causes flashing.  The fix is to
		* register a new window class without these bits and
		* implement special code that damages only the exposed
		* area.
		*/
//		WNDCLASS lpWndClass = new WNDCLASS ();
		CLASSINFO pclsiClassInfo = new CLASSINFO ();
		OS.WinQueryClassInfo (OS.NULLHANDLE, TabFolderClass, pclsiClassInfo);
		TabFolderProc = pclsiClassInfo.pfnWindowProc;
//		TCHAR WC_TABCONTROL = new TCHAR (0, OS.WC_TABCONTROL, true);
//		OS.GetClassInfo (0, WC_TABCONTROL, lpWndClass);
//		TabFolderProc = lpWndClass.lpfnWndProc;
//		int hInstance = OS.GetModuleHandle (null);
//		if (!OS.GetClassInfo (hInstance, TabFolderClass, lpWndClass)) {
//			int hHeap = OS.GetProcessHeap ();
//			lpWndClass.hInstance = hInstance;
//			lpWndClass.style &= ~(OS.CS_HREDRAW | OS.CS_VREDRAW);
//			int byteCount = TabFolderClass.length () * TCHAR.sizeof;
//			int lpszClassName = OS.HeapAlloc (hHeap, OS.HEAP_ZERO_MEMORY, byteCount);
//			OS.MoveMemory (lpszClassName, TabFolderClass, byteCount);
//			lpWndClass.lpszClassName = lpszClassName;
//			OS.RegisterClass (lpWndClass);
//			OS.HeapFree (hHeap, 0, lpszClassName);
//		}
	}

/**
 * Constructs a new instance of this class given its parent
 * and a style value describing its behavior and appearance.
 * <p>
 * The style value is either one of the style constants defined in
 * class <code>SWT</code> which is applicable to instances of this
 * class, or must be built by <em>bitwise OR</em>'ing together 
 * (that is, using the <code>int</code> "|" operator) two or more
 * of those <code>SWT</code> style constants. The class description
 * lists the style constants that are applicable to the class.
 * Style bits are also inherited from superclasses.
 * </p>
 *
 * @param parent a composite control which will be the parent of the new instance (cannot be null)
 * @param style the style of control to construct
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the parent is null</li>
 * </ul>
 * @exception SWTException <ul>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the parent</li>
 *    <li>ERROR_INVALID_SUBCLASS - if this class is not an allowed subclass</li>
 * </ul>
 *
 * @see SWT
 * @see Widget#checkSubclass
 * @see Widget#getStyle
 */
public TabFolder (Composite parent, int style) {
	super (parent, checkStyle (style));
}

/**
 * Adds the listener to the collection of listeners who will
 * be notified when the receiver's selection changes, by sending
 * it one of the messages defined in the <code>SelectionListener</code>
 * interface.
 * <p>
 * When <code>widgetSelected</code> is called, the item field of the event object is valid.
 * <code>widgetDefaultSelected</code> is not called.
 * </p>
 *
 * @param listener the listener which should be notified
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
 * </ul>
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 *
 * @see SelectionListener
 * @see #removeSelectionListener
 * @see SelectionEvent
 */
public void addSelectionListener(SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener(listener);
	addListener(SWT.Selection,typedListener);
	addListener(SWT.DefaultSelection,typedListener);
}

int callWindowProc (int msg, int mp1, int mp2) {
	if (handle == 0) return 0;
        return OS.WinCallWindowProc(TabFolderProc, handle, msg, mp1, mp2);
}

static int checkStyle (int style) {
	/*
	* Even though it is legal to create this widget
	* with scroll bars, they serve no useful purpose
	* because they do not automatically scroll the
	* widget's client area.  The fix is to clear
	* the SWT style.
	*/
	return style & ~(SWT.H_SCROLL | SWT.V_SCROLL);
}

protected void checkSubclass () {
	if (!isValidSubclass ()) error (SWT.ERROR_INVALID_SUBCLASS);
}

//public Point computeSize (int wHint, int hHint, boolean changed) {
//	checkWidget ();
//      RECTL rcl = new RECTL ();
//      RECTL insetRect = new RECTL ();
//      WinQueryWindowPos(hwndDialog, &swpDialog);
//      OS.WinQueryWindowPos(handle, rcl);
//      insetRect.yBottom=insetRect.xLeft=0;
//      insetRect.xRight=rcl.cx;
//      insetRect.yTop=rcl.cy;
//      Return size of notebook page in pixel for given notebook window size (with tab text size set)

//      OS.WinSendMsg(hwndNotebook, BKM_CALCPAGERECT, insetRect, true);
//      Calculate notebook size so that notebook dialog page fits into the notebook page

//      swpNB.cx-=(rectlNB.xRight-rectlNB.xLeft)-ptlNB.x;
//      swpNB.cy-=(rectlNB.yTop-rectlNB.yBottom)-ptlNB.y;
//

//	RECT insetRect = new RECT (), itemRect = new RECT ();
//	OS.SendMessage (handle, OS.TCM_ADJUSTRECT, 0, insetRect);
//	int width = insetRect.left - insetRect.right, height = 0;
//	int count = OS.SendMessage (handle, OS.TCM_GETITEMCOUNT, 0, 0);
//	if (count != 0) {
//		OS.SendMessage (handle, OS.TCM_GETITEMRECT, count - 1, itemRect);
//		width = Math.max (width, itemRect.right - insetRect.right);
//	}
	Point size;
//	if (layout != null) {
//		size = layout.computeSize (this, wHint, hHint, changed);
//	} else {
//		size = minimumSize ();
//	}
//	if (size.x == 0) size.x = DEFAULT_WIDTH;
//	if (size.y == 0) size.y = DEFAULT_HEIGHT;
//	if (wHint != SWT.DEFAULT) size.x = wHint;
//	if (hHint != SWT.DEFAULT) size.y = hHint;
//	width = Math.max (width, size.x);
//	height = Math.max (height, size.y);
//	Rectangle trim = computeTrim (0, 0, width, height);
//	width = trim.width;  height = trim.height;
//	return new Point (width, height);
//}

//public Rectangle computeTrim (int x, int y, int width, int height) {
//	checkWidget ();
//	RECT rect = new RECT ();
//	OS.SetRect (rect, x, y, x + width, y + height);
//	OS.SendMessage (handle, OS.TCM_ADJUSTRECT, 1, rect);
//	int border = getBorderWidth ();
//	rect.left -= border;  rect.right += border;
//	rect.top -= border;  rect.bottom += border;
//	int newWidth = rect.right - rect.left;
//	int newHeight = rect.bottom - rect.top;
//	return new Rectangle (rect.left, rect.top, newWidth, newHeight);
//}

void createItem (TabItem item, int index) {
	int count = OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	if (!(0 <= index && index <= count)) error (SWT.ERROR_INVALID_RANGE);
	if (count == items.length) {
		TabItem [] newItems = new TabItem [items.length + 4];
                int[] newIDs = new int[itemsID.length + 4];
		System.arraycopy (items, 0, newItems, 0, items.length);
                System.arraycopy (itemsID, 0, newIDs, 0, itemsID.length);
		items = newItems;
                itemsID = newIDs;
	}
         int pageID = OS.WinSendMsg (handle, OS.BKM_INSERTPAGE, 0, OS.MPFROM2SHORT((short)(OS.BKA_MAJOR),(short)OS.BKA_LAST));
         //@@INFO(lpino): All tabs are gray to keep a consistent look with the rests of the platforms
         OS.WinSendMsg (handle, OS.BKM_SETTABCOLOR, pageID , 0x00C0C0C0);
         //@@INFO(lpino): The background of empty tabs are set to gray
         OS.WinSendMsg (handle, OS.BKM_SETNOTEBOOKCOLORS, OS.SYSCLR_FIELDBACKGROUND , OS.BKA_BACKGROUNDPAGECOLORINDEX);
	if (pageID == OS.NULLHANDLE) {
		error (SWT.ERROR_ITEM_NOT_ADDED);
	}
	System.arraycopy (items, index, items, index + 1, count - index);
        System.arraycopy (itemsID, index, itemsID, index + 1, count - index);
        itemsID[index] = pageID;
	items [index] = item;
	
	/*
	* Send a selection event when the item that is added becomes
	* the new selection.  This only happens when the first item
	* is added.
	*/
	if (count == 0) {
		Event event = new Event ();
		event.item = items [0];
		sendEvent (SWT.Selection, event);
		// the widget could be destroyed at this point
	}
}

void createHandle () {
	super.createHandle ();
	state &= ~CANVAS;
}

void createWidget () {
	super.createWidget ();
	items = new TabItem [4];
        itemsID = new int[4];
}

void destroyItem (TabItem item) {
	int count = OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	int index = 0;
	while (index < count) {
		if (items [index] == item) break;
		index++;
	}
	if (index == count) return;
        int selectionIndex = itemsID[index];
	if (OS.WinSendMsg (handle, OS.BKM_DELETEPAGE, selectionIndex, (short)OS.BKA_TAB) == 0) {
		error (SWT.ERROR_ITEM_NOT_REMOVED);
	}
	System.arraycopy (items, index + 1, items, index, --count - index);
        System.arraycopy (itemsID, index + 1, itemsID, index, --count - index);
	items [count] = null;
        itemsID[count] = 0;
	if (count == 0) {
//		if (imageList != null) {
//			OS.SendMessage (handle, OS.TCM_SETIMAGELIST, 0, 0);
//			Display display = getDisplay ();
//			display.releaseImageList (imageList);
//		}
//		imageList = null;
		items = new TabItem [4];
                itemsID = new int[4];
	}
	if (count > 0 && index == selectionIndex) {
		setSelection (Math.max (0, selectionIndex - 1));
		selectionIndex = getSelectionIndex ();
		if (selectionIndex != -1) {
			Event event = new Event ();
			event.item = items [selectionIndex];
			sendEvent (SWT.Selection, event);
			// the widget could be destroyed at this point
		}
	}
}

public Rectangle getClientArea () {
	checkWidget ();
	if (parent != null && parent.pswp != 0) {
            endDeferWindowPos (parent);
            int count = parent.getChildrenCount ();
            beginDeferWindowPos (parent, count);
        }
        
	RECTL rect = new RECTL ();
        
        SWP swp = new SWP ();
        OS.WinQueryWindowPos (handle, swp);
                
        rect.yBottom = rect.xLeft = 0;
        rect.xRight = swp.cx;
        rect.yTop = swp.cy;

        OS.WinSendMsg(handle, OS.BKM_CALCPAGERECT, rect, true);
        
	int width = rect.xRight - rect.xLeft;
	int height = rect.yTop - rect.yBottom;
        int y = swp.cy-height;
        
//        System.out.println("XLEFT = " + rect.xLeft);
//        System.out.println("YTOP = " + rect.yTop);
//        System.out.println("XRIGHT = " + rect.xRight);
//        System.out.println("YBOTTOM = " + rect.yBottom);
//        System.out.println("width = " + width);
//        System.out.println("height = " + height);
//        System.out.println("Delta = " + swp.y);

	return new Rectangle (0, y, width, height);
}

/**
 * Returns the item at the given, zero-relative index in the
 * receiver. Throws an exception if the index is out of range.
 *
 * @param index the index of the item to return
 * @return the item at the given index
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_INVALID_RANGE - if the index is not between 0 and the number of elements in the list minus 1 (inclusive)</li>
 * </ul>
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public TabItem getItem (int index) {
	checkWidget ();
	int count =OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	if (!(0 <= index && index < count)) error (SWT.ERROR_INVALID_RANGE);
	return items [index];
}

/**
 * Returns the number of items contained in the receiver.
 *
 * @return the number of items
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public int getItemCount () {
	checkWidget ();
	return OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
}

/**
 * Returns an array of <code>TabItem</code>s which are the items
 * in the receiver. 
 * <p>
 * Note: This is not the actual structure used by the receiver
 * to maintain its list of items, so modifying the array will
 * not affect the receiver. 
 * </p>
 *
 * @return the items in the receiver
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public TabItem [] getItems () {
	checkWidget ();
	int count = OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	TabItem [] result = new TabItem [count];
	System.arraycopy (items, 0, result, 0, count);
	return result;
}

/**
 * Returns an array of <code>TabItem</code>s that are currently
 * selected in the receiver. An empty array indicates that no
 * items are selected.
 * <p>
 * Note: This is not the actual structure used by the receiver
 * to maintain its selection, so modifying the array will
 * not affect the receiver. 
 * </p>
 * @return an array representing the selection
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public TabItem [] getSelection () {
	checkWidget ();
        int pageID = OS.WinSendMsg (handle, OS.BKM_QUERYPAGEID, 0, OS.MPFROM2SHORT((short)OS.BKA_TOP,(short)OS.BKA_MAJOR));
	int index = getTabIndex(pageID);
	if (index == -1) return new TabItem [0];
	return new TabItem [] {items [index]};
}

/**
 * Returns the zero-relative index of the item which is currently
 * selected in the receiver, or -1 if no item is selected.
 *
 * @return the index of the selected item
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public int getSelectionIndex () {
	checkWidget ();
	int pageID = OS.WinSendMsg (handle, OS.BKM_QUERYPAGEID, 0, OS.MPFROM2SHORT((short)OS.BKA_TOP,(short)OS.BKA_MAJOR));
        return getTabIndex(pageID);
}

//int imageIndex (Image image) {
//	if (image == null) return OS.I_IMAGENONE;
//	if (imageList == null) {
//		Rectangle bounds = image.getBounds ();
//		imageList = getDisplay ().getImageList (new Point (bounds.width, bounds.height));
//		int index = imageList.indexOf (image);
//		if (index == -1) index = imageList.add (image);
//		int hImageList = imageList.getHandle ();
//		OS.SendMessage (handle, OS.TCM_SETIMAGELIST, 0, hImageList);
//		return index;
//	}
//	int index = imageList.indexOf (image);
//	if (index != -1) return index;
//	return imageList.add (image);
//}

/**
 * Searches the receiver's list starting at the first item
 * (index 0) until an item is found that is equal to the 
 * argument, and returns the index of that item. If no item
 * is found, returns -1.
 *
 * @param item the search item
 * @return the index of the item
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the string is null</li>
 * </ul>
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public int indexOf (TabItem item) {
	checkWidget ();
	if (item == null) error (SWT.ERROR_NULL_ARGUMENT);
	int count = OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	for (int i=0; i<count; i++) {
		if (items [i] == item) return i;
	}
	return -1;
}

boolean mnemonicHit (char key) {
	for (int i=0; i<items.length; i++) {
		TabItem item = items [i];
		if (item != null) {
			char ch = findMnemonic (item.getText ());
			if (Character.toUpperCase (key) == Character.toUpperCase (ch)) {		
				if (setFocus ()) {
					setSelection (i, true);
					return true;
				}
			}
		}
	}
	return false;
}

boolean mnemonicMatch (char key) {
	for (int i=0; i<items.length; i++) {
		TabItem item = items [i];
		if (item != null) {
			char ch = findMnemonic (item.getText ());
			if (Character.toUpperCase (key) == Character.toUpperCase (ch)) {		
				return true;
			}
		}
	}
	return false;
}

void releaseWidget () {
	int count = OS.WinSendMsg (handle, OS.BKM_QUERYPAGECOUNT, 0, OS.BKA_END);
	for (int i=0; i<count; i++) {
		TabItem item = items [i];
		if (!item.isDisposed ()) item.releaseWidget ();
	}
	items = null;
        itemsID = null;
//	if (imageList != null) {
//		OS.SendMessage (handle, OS.TCM_SETIMAGELIST, 0, 0);
//		Display display = getDisplay ();
//		display.releaseImageList (imageList);
//	}
//	imageList = null;
	super.releaseWidget ();
}

/**
 * Removes the listener from the collection of listeners who will
 * be notified when the receiver's selection changes.
 *
 * @param listener the listener which should no longer be notified
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
 * </ul>
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 *
 * @see SelectionListener
 * @see #addSelectionListener
 */
public void removeSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}

/**
 * Sets the receiver's selection to be the given array of items.
 * The current selected is first cleared, then the new items are
 * selected.
 *
 * @param items the array of items
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public void setSelection (TabItem [] items) {
	checkWidget ();
	if (items == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (items.length == 0) {
		setSelection (-1);
		return;
	}
	for (int i=items.length-1; i>=0; --i) {
		int index = indexOf (items [i]);
		if (index != -1) setSelection (index);
	}
}

/**
 * Selects the item at the given zero-relative index in the receiver. 
 * If the item at the index was already selected, it remains selected.
 * The current selected is first cleared, then the new items are
 * selected. Indices that are out of range are ignored.
 *
 * @param index the index of the item to select
 *
 * @exception SWTException <ul>
 *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
 *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
 * </ul>
 */
public void setSelection (int index) {
	checkWidget ();
	setSelection (index, false);
}

void setSelection (int index, boolean notify) {
	int oldIndex = getSelectionIndex();
	if (oldIndex != -1) {
		TabItem item = items [oldIndex];
		Control control = item.control;
		if (control != null && !control.isDisposed ()) {
			control.setVisible (false);
		}
	}
	OS.WinSendMsg (handle, OS.BKM_TURNTOPAGE, itemsID[index], 0);
	int newIndex = getSelectionIndex();
	if (newIndex != -1) {
		TabItem item = items [newIndex];
		Control control = item.control;
		if (control != null && !control.isDisposed ()) {
			control.setBounds (getClientArea ());
			control.setVisible (true);
		}
		if (notify) {
			Event event = new Event ();
			event.item = item;
			sendEvent (SWT.Selection, event);
		}
	}
}
int getTabIndex(int pageID){
    for(int i=0;i<itemsID.length;i++){
        if(itemsID[i] == pageID)
            return i;
    }
    return -1;
}

int getPageID(int index){
    int firstID = OS.WinSendMsg (handle, OS.BKM_QUERYPAGEID, 0, OS.MPFROM2SHORT((short)OS.BKA_FIRST, (short)OS.BKA_MAJOR));
     int id = firstID;
    for(int i=0;i <= index; i++){
        id = OS.WinSendMsg (handle, OS.BKM_QUERYPAGEID, id, OS.MPFROM2SHORT((short)OS.BKA_NEXT, (short)OS.BKA_MAJOR));
    }
     return id;
}
//int toolTipHandle () {
//	return OS.SendMessage (handle, OS.TCM_GETTOOLTIPS, 0, 0);
//}
//
//String toolTipText (NMTTDISPINFO hdr) {
//	if ((hdr.uFlags & OS.TTF_IDISHWND) != 0) {
//		return null;
//	}
//	int index = hdr.idFrom;
//	int hwndToolTip = toolTipHandle ();
//	if (hwndToolTip == hdr.hwndFrom) {
//		if (0 <= index && index < items.length) {
//			TabItem item = items [index];
//			if (item != null) return item.toolTipText;
//		}
//	}
//	return super.toolTipText (hdr);
//}

boolean traversePage (boolean next) {
	int count = getItemCount ();
	if (count == 0) return false;
	int index = getSelectionIndex ();
	if (index == -1) {
		index = 0;
	} else {
		int offset = (next) ? 1 : -1;
		index = (index + offset + count) % count;
	}
	setSelection (index, true);
	return index == getSelectionIndex ();
}

int widgetStyle () {
	int bits = super.widgetStyle () | OS.WS_VISIBLE | OS.WS_CLIPCHILDREN | OS.BKS_TABBEDDIALOG | OS.BKS_MAJORTABTOP | OS.WS_CLIPSIBLINGS;	
	return bits;
}

PSZ windowClass () {
	return TabFolderClass;
}

int windowProc () {
	return TabFolderProc;
}

boolean setTabDimensions( PSZ tabText ){
        int iSize = 0;
        int hps = this.hps;
        if (hps == 0) hps = OS.WinGetPS (handle);
        FONTMETRICS fm = new FONTMETRICS();
        if(OS.GpiQueryFontMetrics (hps, FONTMETRICS.sizeof, fm))
            fm.lMaxBaselineExt<<=1;
        else
            fm.lMaxBaselineExt = 30;

        iSize = getStringSize( hps, tabText );

        if( iSize > iLongestMajText )
            iLongestMajText = iSize;

        if (this.hps == 0) OS.WinReleasePS (hps);

        OS.WinSendMsg(handle, OS.BKM_SETDIMENSIONS,
                  OS.MPFROM2SHORT((short)(iLongestMajText+5), (short)((float)fm.lMaxBaselineExt * 0.8)),
                  (short)OS.BKA_MAJORTAB);

       OS.WinSendMsg(handle, OS.BKM_SETDIMENSIONS, OS.MPFROM2SHORT((short)0, (short)0),(short)OS.BKA_MINORTAB);
  return true;
}

int getStringSize( int hps, PSZ szString )
{
    int[] pnts = new int [OS.TXTBOX_COUNT * 2];
    OS.GpiQueryTextBox( hps, szString.getBytes().length, szString.getBytes(), OS.TXTBOX_COUNT, pnts);
    return pnts[8];
}


//LRESULT WM_GETDLGCODE (int wParam, int lParam) {
//	LRESULT result = super.WM_GETDLGCODE (wParam, lParam);
//	/*
//	* Return DLGC_BUTTON so that mnemonics will be
//	* processed without needing to press the ALT key
//	* when the widget has focus.
//	*/
//	if (result != null) return result;
//	return new LRESULT (OS.DLGC_BUTTON);
//}
//
//LRESULT WM_NCHITTEST (int wParam, int lParam) {
//	LRESULT result = super.WM_NCHITTEST (wParam, lParam);
//	if (result != null) return result;
//	/*
//	* Feature in Windows.  The tab control implements
//	* WM_NCHITTEST to return HTCLIENT when the cursor
//	* is inside the tab buttons.  This causes mouse
//	* events like WM_MOUSEMOVE to be delivered to the
//	* parent.  Also, tool tips for the tab control are
//	* never invoked because tool tips rely on mouse
//	* events to be delivered to the window that wants
//	* to display the tool tip.  The fix is to call the
//	* default window proc that returns HTCLIENT when
//	* the mouse is in the client area.	
//	*/
//	int hittest = OS.DefWindowProc (handle, OS.WM_NCHITTEST, wParam, lParam);
//	return new LRESULT (hittest);
//}
//
MRESULT WM_SIZE (int mp1, int mp2) {
	MRESULT result = super.WM_SIZE (mp1, mp2);
//	/*
//	* It is possible (but unlikely), that application
//	* code could have disposed the widget in the resize
//	* event.  If this happens, end the processing of the
//	* Windows message by returning the result of the
//	* WM_SIZE message.
//	*/
//	if (isDisposed ()) return result;
//	int index = OS.SendMessage (handle, OS.TCM_GETCURSEL, 0, 0);
//	if (index != -1) {
//		TabItem item = items [index];
//		Control control = item.control;
//		if (control != null && !control.isDisposed ()) {
//			control.setBounds (getClientArea ());
//		}
//	}
	return result;
}

//LRESULT WM_WINDOWPOSCHANGING (int wParam, int lParam) {
//	LRESULT result = super.WM_WINDOWPOSCHANGING (wParam, lParam);
//	if (result != null) return result;
//	if (!OS.IsWindowVisible (handle)) return result;
//	WINDOWPOS lpwp = new WINDOWPOS ();
//	OS.MoveMemory (lpwp, lParam, WINDOWPOS.sizeof);
//	if ((lpwp.flags & (OS.SWP_NOSIZE | OS.SWP_NOREDRAW)) != 0) {
//		return result;
//	}
//	int bits = OS.GetWindowLong (handle, OS.GWL_STYLE);
//	if ((bits & OS.TCS_MULTILINE) != 0) {
//		OS.InvalidateRect (handle, null, true);
//		return result;
//	}
//	RECT rect = new RECT ();
//	OS.SetRect (rect, 0, 0, lpwp.cx, lpwp.cy);
//	OS.SendMessage (handle, OS.WM_NCCALCSIZE, 0, rect);
//	int newWidth = rect.right - rect.left;
//	int newHeight = rect.bottom - rect.top;
//	OS.GetClientRect (handle, rect);
//	int oldWidth = rect.right - rect.left;
//	int oldHeight = rect.bottom - rect.top;
//	if ((newWidth == oldWidth) && (newHeight == oldHeight)) {
//		return result;
//	}
//	RECT inset = new RECT ();
//	OS.SendMessage (handle, OS.TCM_ADJUSTRECT, 0, inset);
//	int marginX = -inset.right, marginY = -inset.bottom;
//	if (newWidth != oldWidth) {
//		int left = oldWidth;
//		if (newWidth < oldWidth) left = newWidth;
//		OS.SetRect (rect, left - marginX, 0, newWidth, newHeight);
//		OS.InvalidateRect (handle, rect, true);
//	}
//	if (newHeight != oldHeight) {
//		int bottom = oldHeight;
//		if (newHeight < oldHeight) bottom = newHeight;
//		if (newWidth < oldWidth) oldWidth -= marginX;
//		OS.SetRect (rect, 0, bottom - marginY, oldWidth, newHeight);
//		OS.InvalidateRect (handle, rect, true);
//	}
//	return result;
//}
//
MRESULT wmControlChild (int mp1, int mp2) {
        int code = mp1 >> 16;
	switch (code) {
		case OS.BKN_PAGESELECTED:                    
		case OS.BKN_PAGESELECTEDPENDING:
			TabItem item = null;
			int pageID = OS.WinSendMsg (handle, OS.BKM_QUERYPAGEID, 0, OS.MPFROM2SHORT((short)OS.BKA_TOP,(short)OS.BKA_MAJOR));
                        int index = getTabIndex(pageID);
			if (index != -1) item = items [index];
			if (item != null) {
				Control control = item.control;
				if (control != null && !control.isDisposed ()) {
					if (code == OS.BKN_PAGESELECTED) {
						control.setBounds (getClientArea ());
					}
					control.setVisible (code == OS.BKN_PAGESELECTED);
				}
			}
			if (code == OS.BKN_PAGESELECTED) {
				Event event = new Event ();
				event.item = item;
				postEvent (SWT.Selection, event);
			}
//                        break;
//            case OS.BKN_NEWPAGESIZE:
//                System.out.println("resize");
//                break;
         //	if (isDisposed ()) return result;
        //	int index = OS.SendMessage (handle, OS.TCM_GETCURSEL, 0, 0);
        //	if (index != -1) {
        //		TabItem item = items [index];
        //		Control control = item.control;
        //		if (control != null && !control.isDisposed ()) {
        //			control.setBounds (getClientArea ());
        //		}
        //	}

	}
	return super.wmControlChild (mp1, mp2);
}
}
